using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using DPC.APC.Plugins.SDK;
using Microsoft.Extensions.Logging;

namespace SupplementaryAreaPlugin.Actions;

/// <summary>
/// Handles sending notifications for supplementary area changes
/// </summary>
internal sealed class SupplementaryNotificationService
{
    private readonly IAppAccess app;
    private readonly ILogger logger;

    public SupplementaryNotificationService(IAppAccess app, ILogger logger)
    {
        this.app = app ?? throw new ArgumentNullException(nameof(app));
        this.logger = logger ?? throw new ArgumentNullException(nameof(logger));
    }

    /// <summary>
    /// Send all notifications for detected changes
    /// </summary>
    public async Task SendNotificationsAsync(
        string recordId,
        ModelChanges? changes,
        CancellationToken ct)
    {
        if (changes == null)
        {
            return;
        }

        // Send notifications for new people assignments
        await NotifyNewPeopleAssignmentsAsync(recordId, changes.NewPeopleAssignments, ct)
            .ConfigureAwait(false);

        // Send notifications for completed endorsements
        await NotifyEndorsementCompletionsAsync(recordId, changes.EndorsementCompletions, ct)
            .ConfigureAwait(false);

        // Send key details changed notifications (for comments and endorsement changes)
        await SendKeyDetailsChangedNotificationAsync(recordId, changes.KeyDetailsChanges, ct)
            .ConfigureAwait(false);
    }

    private async Task NotifyNewPeopleAssignmentsAsync(
        string recordId,
        List<PersonAssignment> assignments,
        CancellationToken ct)
    {
        if (assignments.Count == 0)
        {
            return;
        }

        // Group assignments by notification type
        var leaderAssignments = assignments.Where(a => a.NotificationType == NotificationType.LeaderAssigned).ToList();
        var contributorAssignments = assignments.Where(a => a.NotificationType == NotificationType.ContributorAssigned).ToList();

        logger.LogInformation("Sending {LeaderCount} leader and {ContributorCount} contributor assignment notifications for record {RecordId}",
            leaderAssignments.Count, contributorAssignments.Count, recordId);

        // Send leader notifications (Executive Directors and Allocators)
        foreach (var assignment in leaderAssignments)
        {
            try
            {
                await SendLeaderAssignmentNotificationAsync(recordId, assignment, ct)
                    .ConfigureAwait(false);
            }
            catch (Exception ex)
            {
                logger.LogWarning(ex,
                    "Failed to send leader assignment notification to {Person} for area {AreaIndex}",
                    assignment.Person.DisplayName ?? assignment.Person.Mail,
                    assignment.AreaIndex);
            }
        }

        // Send contributor notifications
        foreach (var assignment in contributorAssignments)
        {
            try
            {
                await SendContributorAssignmentNotificationAsync(recordId, assignment, ct)
                    .ConfigureAwait(false);
            }
            catch (Exception ex)
            {
                logger.LogWarning(ex,
                    "Failed to send contributor assignment notification to {Person} for area {AreaIndex}",
                    assignment.Person.DisplayName ?? assignment.Person.Mail,
                    assignment.AreaIndex);
            }
        }
    }


    private async Task SendLeaderAssignmentNotificationAsync(
        string recordId,
        PersonAssignment assignment,
        CancellationToken ct)
    {
        if (string.IsNullOrWhiteSpace(assignment.Person.Mail))
        {
            logger.LogWarning("Cannot send leader assignment notification - recipient has no email address");
            return;
        }

        var parameters = new
        {
            recordId,
            pluginRelativeTemplatePath = SupplementaryAreasConstants.EmailTemplateLeaderAssigned,
            recipients = new[]
            {
                new
                {
                    graphUserId = assignment.Person.GraphUserId,
                    displayName = assignment.Person.DisplayName,
                    mail = assignment.Person.Mail,
                    jobTitle = assignment.Person.JobTitle,
                    userPrincipalName = assignment.Person.UserPrincipalName
                }
            },
            subject = $"You've been assigned as {assignment.Role}",
            tokens = new
            {
                role = assignment.Role,
                roleField = assignment.RoleField,
                areaIndex = assignment.AreaIndex.ToString(),
                areaLabel = assignment.AreaLabel,
            }
        };

        var parametersJson = JsonSerializer.Serialize(parameters);
        logger.LogDebug("Sending leader assignment notification to {Email} for area {AreaIndex}",
            assignment.Person.Mail, assignment.AreaIndex);

        var response = await app.UniversalAppAccess.InvokeActionAsync(
            SupplementaryAreasConstants.UniversalActionSendNotification,
            parametersJson,
            ct).ConfigureAwait(false);

        CheckNotificationResponse(response, "leader assignment");
    }

    private async Task SendContributorAssignmentNotificationAsync(
        string recordId,
        PersonAssignment assignment,
        CancellationToken ct)
    {
        if (string.IsNullOrWhiteSpace(assignment.Person.Mail))
        {
            logger.LogWarning("Cannot send contributor assignment notification - recipient has no email address");
            return;
        }

        var parameters = new
        {
            recordId,
            pluginRelativeTemplatePath = SupplementaryAreasConstants.EmailTemplateContributorAssigned,
            recipients = new[]
            {
                new
                {
                    graphUserId = assignment.Person.GraphUserId,
                    displayName = assignment.Person.DisplayName,
                    mail = assignment.Person.Mail,
                    jobTitle = assignment.Person.JobTitle,
                    userPrincipalName = assignment.Person.UserPrincipalName
                }
            },
            subject = $"You've been assigned as {assignment.Role}",
            tokens = new
            {
                role = assignment.Role,
                roleField = assignment.RoleField,
                areaIndex = assignment.AreaIndex.ToString(),
                areaLabel = assignment.AreaLabel,
            }
        };

        var parametersJson = JsonSerializer.Serialize(parameters);
        logger.LogDebug("Sending contributor assignment notification to {Email} for area {AreaIndex}",
            assignment.Person.Mail, assignment.AreaIndex);

        var response = await app.UniversalAppAccess.InvokeActionAsync(
            SupplementaryAreasConstants.UniversalActionSendNotification,
            parametersJson,
            ct).ConfigureAwait(false);

        CheckNotificationResponse(response, "contributor assignment");
    }

    private async Task NotifyEndorsementCompletionsAsync(
        string recordId,
        List<EndorsementCompletion> completions,
        CancellationToken ct)
    {
        if (completions.Count == 0) return;

        logger.LogInformation("Sending {Count} endorsement completion notifications for record {RecordId}",
            completions.Count, recordId);

        foreach (var completion in completions)
        {
            try
            {
                await SendEndorsementCompletionNotificationAsync(recordId, completion, ct)
                    .ConfigureAwait(false);
            }
            catch (Exception ex)
            {
                logger.LogWarning(ex,
                    "Failed to send endorsement completion notification for area {AreaIndex}",
                    completion.AreaIndex);
            }
        }
    }

    private async Task SendEndorsementCompletionNotificationAsync(
        string recordId,
        EndorsementCompletion completion,
        CancellationToken ct)
    {
        // Get Lead Author from record to notify
        var leadAuthor = await GetLeadAuthorFromRecordAsync(recordId, ct).ConfigureAwait(false);
        if (leadAuthor == null || string.IsNullOrWhiteSpace(leadAuthor.Mail))
        {
            logger.LogWarning("Cannot send endorsement notification - Lead Author not found or has no email for record {RecordId}", recordId);
            return;
        }

        var parameters = new
        {
            recordId,
            pluginRelativeTemplatePath = SupplementaryAreasConstants.EmailTemplateEndorsementCompleted,
            recipients = new[]
            {
                new
                {
                    graphUserId = leadAuthor.GraphUserId,
                    displayName = leadAuthor.DisplayName,
                    mail = leadAuthor.Mail,
                    jobTitle = leadAuthor.JobTitle,
                    userPrincipalName = leadAuthor.UserPrincipalName
                }
            },
            subject = $"Supplementary Area {completion.AreaIndex} Endorsed",
            tokens = new
            {
                areaIndex = completion.AreaIndex.ToString(),
                areaLabel = completion.AreaLabel,
            }
        };

        var parametersJson = JsonSerializer.Serialize(parameters);

        logger.LogDebug("Sending endorsement completion notification to Lead Author {Email} for area {AreaIndex}", 
            leadAuthor.Mail, completion.AreaIndex);

        var response = await app.UniversalAppAccess.InvokeActionAsync(
            SupplementaryAreasConstants.UniversalActionSendNotification,
            parametersJson,
            ct).ConfigureAwait(false);

        CheckNotificationResponse(response, "endorsement completion");
    }

    private async Task<SupplementaryAreasPlugin.Person?> GetLeadAuthorFromRecordAsync(string recordId, CancellationToken ct)
    {
        try
        {
            var recordSummary = await app.Records.GetRecordAsync(recordId, ct).ConfigureAwait(false);
            if (recordSummary == null)
            {
                logger.LogWarning("Record not found: {RecordId}", recordId);
                return null;
            }

            // Get Lead Author from custom fields
            if (recordSummary.CustomFieldValues != null && 
                recordSummary.CustomFieldValues.TryGetValue(SupplementaryAreasConstants.FieldNameLeadAuthor, out var leadAuthorJson))
            {
                if (!string.IsNullOrWhiteSpace(leadAuthorJson))
                {
                    try
                    {
                        using var doc = JsonDocument.Parse(leadAuthorJson);
                        var root = doc.RootElement;
                        
                        if (root.ValueKind == JsonValueKind.Array && root.GetArrayLength() > 0)
                        {
                            // People picker field returns array, get first person
                            var firstPerson = root[0];
                            return ParsePersonFromJson(firstPerson);
                        }
                        else if (root.ValueKind == JsonValueKind.Object)
                        {
                            return ParsePersonFromJson(root);
                        }
                    }
                    catch (Exception ex)
                    {
                        logger.LogWarning(ex, "Failed to parse Lead Author for record {RecordId}", recordId);
                    }
                }
            }

            logger.LogWarning("Lead Author not found in record {RecordId}", recordId);
            return null;
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Failed to get Lead Author from record {RecordId}", recordId);
            return null;
        }
    }

    private SupplementaryAreasPlugin.Person? ParsePersonFromJson(JsonElement personEl)
    {
        try
        {
            var person = new SupplementaryAreasPlugin.Person
            {
                GraphUserId = GetStringProperty(personEl, "GraphUserId") ?? GetStringProperty(personEl, "id"),
                DisplayName = GetStringProperty(personEl, "DisplayName") ?? GetStringProperty(personEl, "displayName"),
                JobTitle = GetStringProperty(personEl, "JobTitle") ?? GetStringProperty(personEl, "jobTitle"),
                Mail = GetStringProperty(personEl, "Mail") ?? GetStringProperty(personEl, "mail"),
                UserPrincipalName = GetStringProperty(personEl, "UserPrincipalName") ?? GetStringProperty(personEl, "userPrincipalName")
            };

            // Validate we have at least an email
            if (string.IsNullOrWhiteSpace(person.Mail))
            {
                return null;
            }

            return person;
        }
        catch
        {
            return null;
        }
    }

    private string? GetStringProperty(JsonElement element, string propertyName)
    {
        if (element.TryGetProperty(propertyName, out var prop) && prop.ValueKind == JsonValueKind.String)
        {
            return prop.GetString();
        }
        return null;
    }

    private void CheckNotificationResponse(string response, string notificationType)
    {
        try
        {
            using var doc = JsonDocument.Parse(response);
            if (doc.RootElement.TryGetProperty("error", out var errorEl))
            {
                var errorCode = errorEl.GetString();
                logger.LogWarning("Notification failed for {Type}: {Error}", notificationType, errorCode);
            }
            else if (doc.RootElement.TryGetProperty("queuedCount", out var countEl))
            {
                var count = countEl.GetInt32();
                logger.LogDebug("Notification queued successfully for {Type}: {Count} recipients", 
                    notificationType, count);
            }
        }
        catch (Exception ex)
        {
            logger.LogTrace(ex, "Could not parse notification response (non-critical)");
        }
    }

    /// <summary>
    /// Sends key details changed notification for comments and endorsement changes
    /// </summary>
    private async Task SendKeyDetailsChangedNotificationAsync(
        string recordId,
        Dictionary<string, string> keyDetailsChanges,
        CancellationToken ct)
    {
        if (keyDetailsChanges == null || keyDetailsChanges.Count == 0)
        {
            return;
        }

        logger.LogInformation(
            "Sending key details changed notification for record {RecordId} with {Count} changes",
            recordId,
            keyDetailsChanges.Count);

        try
        {
            var parameters = new
            {
                recordId,
                changes = keyDetailsChanges
            };

            var parametersJson = JsonSerializer.Serialize(parameters);

            logger.LogDebug("Sending key details changed notification: {Params}", parametersJson);

            var response = await app.UniversalAppAccess.InvokeActionAsync(
                SupplementaryAreasConstants.UniversalActionSendKeyDetailsChangedNotification,
                parametersJson,
                ct).ConfigureAwait(false);

            CheckKeyDetailsNotificationResponse(response, keyDetailsChanges.Count);
        }
        catch (Exception ex)
        {
            logger.LogWarning(ex,
                "Failed to send key details changed notification for record {RecordId}",
                recordId);
        }
    }

    private void CheckKeyDetailsNotificationResponse(string response, int changeCount)
    {
        try
        {
            using var doc = JsonDocument.Parse(response);
            if (doc.RootElement.TryGetProperty("error", out var errorEl))
            {
                var errorCode = errorEl.GetString();
                logger.LogWarning(
                    "Key details changed notification failed: {Error}",
                    errorCode);
            }
            else if (doc.RootElement.TryGetProperty("status", out var statusEl))
            {
                var status = statusEl.GetString();
                logger.LogDebug(
                    "Key details changed notification completed with status {Status} for {Count} changes",
                    status,
                    changeCount);
            }
        }
        catch (Exception ex)
        {
            logger.LogTrace(ex, "Could not parse key details notification response (non-critical)");
        }
    }
}

